#include "Camera.h"

Camera::Camera(float speed, Vector3 origin, Vector3 pitchYawRoll)
{
	//Default values
	_Speed = speed;
	_OriginPosition = origin;
	_PitchYawRoll = pitchYawRoll;
	_MinimumBounds = Vector3(-1000.0f, -1000.0f, 1000.0f);
	_MaximumBounds = Vector3(1000.0f, 1000.0f, -1000.0f);
}

Camera::~Camera()
{
}

void Camera::Animate(float dt)
{

}

void Camera::Update()
{
	//Paul's calculations
	float cosR, cosP, cosY;
	float sinR, sinP, sinY;
	cosY = cosf(_Yaw * 3.1415f / 180);
	cosP = cosf(_Pitch * 3.1415f / 180);
	cosR = cosf(_Roll * 3.1415f / 180);
	sinY = sinf(_Yaw * 3.1415f / 180);
	sinP = sinf(_Pitch * 3.1415f / 180);
	sinR = sinf(_Roll * 3.1415f / 180);

	ResetToBounds();	//Reset the camera back into the viewing bounds if it tries to go past

	//Vector calculations
	_ForwardVector = Vector3(sinY * cosP, sinP, cosP * -cosY);
	_LookAtVector = _PositionVector + _ForwardVector;
	_UpVector = Vector3(-cosY * sinR - sinY * sinP * cosR, cosP * cosR, -sinY * sinR - sinP * cosR * -cosY);
	_RightVector = Vector3(
		((_ForwardVector.y * _UpVector.z) - (_ForwardVector.z * _UpVector.y)), -
		((_ForwardVector.x * _UpVector.z) - (_ForwardVector.z * _UpVector.x)),
		((_ForwardVector.x * _UpVector.y) - (_ForwardVector.y * _UpVector.x))
	);
}

void Camera::ResetToBounds()
{
	if (!DebugDisableBounds)	//If we've got bounds enabled
	{
		//Determine which axis we've gone past the bounds of and then set the position vector
		if (_PositionVector.x > _MaximumBounds.x)
		{
			_PositionVector.x = _MaximumBounds.x;
		}

		if (_PositionVector.y > _MaximumBounds.y)
		{
			_PositionVector.y = _MaximumBounds.y;
		}

		if (_PositionVector.z < _MaximumBounds.z)
		{
			_PositionVector.z = _MaximumBounds.z;
		}

		if (_PositionVector.x < _MinimumBounds.x)
		{
			_PositionVector.x = _MinimumBounds.x;
		}

		if (_PositionVector.y < _MinimumBounds.y)
		{
			_PositionVector.y = _MinimumBounds.y;
		}

		if (_PositionVector.z > _MinimumBounds.z)
		{
			_PositionVector.z = _MinimumBounds.z;
		}
	}
}

Vector3 Camera::GetPosition()
{
	return _PositionVector;
}

Vector3 Camera::GetLookAt()
{
	return _LookAtVector;
}

Vector3 Camera::GetUp()
{
	return _UpVector;
}

Vector3 Camera::GetPitchYawRoll()
{
	return Vector3(_Pitch, _Yaw, _Roll);
}

/*Vector3 Camera::DebugGetForward()
{
	return _ForwardVector;
}

Vector3 Camera::DebugGetLookAt()
{
	return _LookAtVector;
}

Vector3 Camera::DebugGetRight()
{
	return _RightVector;
}*/

void Camera::Reset()
{
	//Put the vectors back to their defaults and set the axis values to their defaults, then call Update to reflect reset changes
	_PositionVector = Vector3(_OriginPosition.x, _OriginPosition.y, _OriginPosition.z);
	_LookAtVector = Vector3(0.0f, 0.0f, 0.0f);
	_UpVector = Vector3(0.0f, 1.0f, 0.0f);
	_Pitch = _PitchYawRoll.x;
	_Yaw = _PitchYawRoll.y;
	_Roll = _PitchYawRoll.z;
	Update();
}

void Camera::SetBounds(Vector3 minBounds, Vector3 maxBounds)
{
	//Sets new bounds values
	_MinimumBounds = minBounds;
	_MaximumBounds = maxBounds;

}

void Camera::SetOrigins(Vector3 origin, Vector3 pitchYawRoll)
{
	//Set the camera's origin values aka where it should reset to
	_OriginPosition = origin;
	_PitchYawRoll = pitchYawRoll;
}

void Camera::MoveForward(float dt)
{
	_PositionVector.x += _ForwardVector.x * _Speed * dt;
	_PositionVector.y += _ForwardVector.y * _Speed * dt;
	_PositionVector.z += _ForwardVector.z * _Speed * dt;
	Update();
}

void Camera::MoveRight(float dt)
{
	_PositionVector.x += _RightVector.x * _Speed * dt;
	_PositionVector.y += _RightVector.y * _Speed * dt;
	_PositionVector.z += _RightVector.z * _Speed * dt;
	Update();
}

void Camera::MoveUp(float dt)
{
	_PositionVector.y += _Speed * dt;
	Update();
}

void Camera::RotateX(float dt)
{
	_Pitch += _MouseRotationSpeed * dt;
	Update();
}

void Camera::RotateY(float dt)
{
	_Yaw += _MouseRotationSpeed * dt;
	Update();
}

void Camera::RotateZ(float dt)
{
	_Roll += _MouseRotationSpeed * dt;
	Update();
}

void Camera::RotateFromMouse(float xOffset, float yOffset, float dt)
{
	if (xOffset > 0)
	{
		_Yaw += _MouseRotationSpeed * dt * log(abs(xOffset));
		_Yaw = fmod(_Yaw, 360);	//Clamping

		if (_Yaw < 0)
		{
			_Yaw += 360;
		}
	}

	else if (xOffset < 0)
	{
		_Yaw += _MouseRotationSpeed * dt * log(abs(xOffset)) * -1;
		_Yaw = fmod(_Yaw, 360);	//Clamping
		
		if (_Yaw < 0)
		{
			_Yaw += 360;
		}
	}

	if (yOffset > 0)
	{
		_Pitch += _MouseRotationSpeed * dt * log(abs(yOffset)) * -1;
	}

	else if (yOffset < 0)
	{
		_Pitch += _MouseRotationSpeed * dt * log(abs(yOffset));
	}

	Update();
}

bool Camera::IsFacing(Direction direction)
{
	//Checks if we're facing direction
	switch (direction)
	{
		case Direction::North:
			if (_Yaw <= 45 || _Yaw > 315)	// 316 - 45
			{
				return true;
			}
			break;

		case Direction::East:
			if (_Yaw <= 135 && _Yaw > 45)	// 46 - 135
			{
				return true;
			}
			break;

		case Direction::South:
			if (_Yaw <= 225 && _Yaw > 135)	//136 - 225
			{
				return true;
			}
			break;

		case Direction::West:
			if (_Yaw <= 315 && _Yaw > 225)	//226 - 315
			{
				return true;
			}
			break;
	}

	return false;
}


